## to set wd as current directory:
# setwd(dirname(rstudioapi::getActiveDocumentContext()$path))

# Required Libraries ------------------------------------------------
require(classmap)     # makeFV
require(ggplot2)      # plots
require(kernlab)      # kernel functions
require(scales)       # ggplot coloring
require(gridExtra)    # ggplot
require(mltools)      # calculating mcc (can be replaced with custom code)
require(dplyr)        # pipe operations %>%
require(pcaPP)        # for l1 median in KOD algorithm pcaPP::l1median_NLM
require(tictoc)       # time tracking
require(ddalpha)      # computing exact SDO
require(matrixStats)  # colVars
require(robustbase)   # Qn and M estimator
require(stats)        # for dist and median
require(dbscan)       # KNN and LOF
require(isotree)      # Isolation Forest
require(e1071)        # SVM


# Utilities for Data Generation --------------------------------------

# Generate Moons data, taken from clusteringdatasets package
make_moons <- function(n_samples = 100, shuffle = TRUE, noise = NA) {
  n_samples_out <- trunc(n_samples / 2)
  n_samples_in <- n_samples - n_samples_out

  points <- matrix( c(
    cos(seq(from = 0, to = pi, length.out = n_samples_out)),  # Outer circle x
    1 - cos(seq(from = 0, to = pi, length.out = n_samples_in)), # Inner circle x
    sin(seq(from = 0, to = pi, length.out = n_samples_out)), # Outer circle y
    1 - sin(seq(from = 0, to = pi, length.out = n_samples_in)) - 0.5 # Inner circle y
  ), ncol = 2)

  if (!is.na(noise)) points <- points + rnorm(length(points), sd = noise)

  labels <- c(rep(1, n_samples_out), rep(2, n_samples_in))

  if (!shuffle) {
    list(
      samples = points,
      labels = labels
    )
  } else {
    order <- sample(x = n_samples, size = n_samples, replace = F)
    list(
      samples = points[order,],
      labels = labels[order]
    )
  }
}

# Generates a circle with outliers clustered around the its center
cans_circle <- function(nclean, p.cont, rad = 1, noise = 0.05,
                        radcluster = 0.3, seed = 42) {
  set.seed(seed)
  # Generating the unit circle class
  angles_circle <- runif(nclean, 0, 2*pi)
  x_circle <- rad * cos(angles_circle) + rnorm(nclean, 0, noise)
  y_circle <- rad * sin(angles_circle) + rnorm(nclean, 0, noise)

  # Generate cluster
  noutlier <- floor(nclean * p.cont / (1 - p.cont))
  angles_cluster <- runif(noutlier, 0, 2*pi)
  radii_cluster <- runif(noutlier, 0, radcluster)
  x_cluster <- radii_cluster * cos(angles_cluster)
  y_cluster <- radii_cluster * sin(angles_cluster)

  data <- rbind(cbind(x_circle, y_circle), cbind(x_cluster, y_cluster))
  colnames(data) <- c("X", "Y")
  return(data)
}


# Generates a ring with SALT and PEPPER noise
cans_saltpepper <- function(nclean, p.cont, rad = 1, noise = 0.05,
                            border = 0.3, sp.mult = 2, seed = 42) {
  set.seed(seed)
  # Generating the unit ring class
  angles_circle <- runif(nclean, 0, 2*pi)
  x_circle <- rad * cos(angles_circle) + rnorm(nclean, 0, noise)
  y_circle <- rad * sin(angles_circle) + rnorm(nclean, 0, noise)

  # Generate salt pepper
  noutlier = floor(nclean * p.cont / (1 - p.cont))
  lim = sp.mult * rad
  x_outlier = runif(noutlier, -lim, lim)
  y_outlier = runif(noutlier, -lim, lim)

  # Push outliers near the ring
  r = sqrt(x_outlier^2 + y_outlier^2)
  weak = which(rad * (1 - border) < r & r < (1 + border) * rad)
  while (length(weak)) {
    x_outlier[weak] = runif(length(weak), -lim, lim)
    y_outlier[weak] = runif(length(weak), -lim, lim)
    r = sqrt(x_outlier^2 + y_outlier^2)
    weak = which(rad * (1 - border) < r & r < (1 + border) * rad)
  }

  data = rbind(cbind(x_circle, y_circle), cbind(x_outlier, y_outlier))
  colnames(data) = c("X", "Y")
  return(data)
}


# inside-outside data
# ntotal is fixed to 1000
# hardcoded just for reproducibility
cans_insideoutside <- function(p.cont, rad = 2, noise = 0.2,
                               radcluster = 0, seed = 42) {
  if (p.cont == 0.05) {
    nclean = 950
    nclean2 = 25
    p.temp = 0.026
  } else if (p.cont == 0.10) {
    nclean = 900
    nclean2 = 50
    p.temp = 0.053
  } else if (p.cont == 0.20) {
    nclean = 800
    nclean2 = 100
    p.temp = 0.112
  } else {
    stop("p.cont must be one of: 0.5, 0.10, or 0.20")
  }
  circle_in = cans_circle(nclean = nclean, p.cont = p.temp, seed = seed)
  circle_out = cans_circle(nclean = nclean2, p.cont = 0, rad = 2, noise = 0.2, radcluster = 0, seed = seed)
  data = rbind(circle_in, circle_out)
  colnames(data) = c("X", "Y")
  return(data)
}


# Local Outlier Factor -----------------------------------------------

lof_pmax <- function(dm, minPts_values) {
  res = rep(0, nrow(dm))
  for (mpts in minPts_values) {
    tmpres <- dbscan::lof(dm, minPts = mpts)
    res = pmax(res, tmpres)
  }
  return(res)
}


# Utilities for KOD ---------------------------------------------------

# GENERATE UNIT DIRECTIONS
generate_unit_directions <- function(x,
                                     type,
                                     num_dir = NULL,
                                     center_for_point = "L1-median",
                                     seed = NULL) {
  # This function generates unit directions for the KOD algorithm
  #
  # x                 : data matrix
  # type              : type of directions to generate
  # num_dir           : number of directions to generate (ignored for 
  #                     point and basis vector types)
  # center_for_point  : center for point directions (ignored for
  #                     other direction types)
  # seed              : seed for reproducibility

  # Generate seed if not provided
  if (is.null(seed)) {
    seed = ceiling(runif(1, 0, 1) * 10000)
    message(paste("Seed is not provided. Generated seed is", seed))
  }

  # Check if number of directions is given
  if (is.null(num_dir)) {
    if (type == "two_points" | type == "random") {
      stop("num_dir must be specified for two_points and random types")
    }
  }

  # Point type
  if (type == "point") {
    if (center_for_point == "coordwise median") {
      center <- apply(x, 2, median)
    } else if (center_for_point == "L1-median") {
      if (ncol(x) == 1) {
        center <- median(x)
      } else {
        center <- pcaPP::l1median_NLM(x, MaxStep = 2000, ItTol = 1e-08, 
                                      trace = F, m.init = colMeans(x))$par
      }
    } else {
      stop("center_for_point not recognized")
    }
    unit_dirs <- t(x) - center
    unit_dirs <- apply(unit_dirs, 2, function(x) {
      tempnorm <- norm(x, type = "2")
      if (tempnorm == 0) { return(rep(0, length(x))) }
      else return(x / tempnorm)
    })
  }

  # Two points type
  if (type == "two_points") {
    n <- nrow(x)
    all_pairs <- combn(1:n, 2)
    if (num_dir > ncol(all_pairs)) {
      num_dir <- ncol(all_pairs)
    }
    # Randomly select num_dir pairs
    set.seed(seed)
    selected_pairs <- sample(1:ncol(all_pairs), num_dir, replace = F)
    unit_dirs <- t(x[all_pairs[1, selected_pairs], ] - 
                     x[all_pairs[2, selected_pairs], ])
    unit_dirs <- apply(unit_dirs, 2, function(x) {
      tempnorm <- norm(x, type = "2")
      if (tempnorm == 0) { return(rep(0, length(x))) }
      else return(x / tempnorm)
    })
  }


  # Basis Vector type
  if (type == "basis_vector") {
    unit_dirs <- diag(ncol(x))
  }

  # Random type
  if (type == "random") {
    set.seed(seed)
    unit_dirs <- matrix(rnorm(ncol(x) * num_dir), nrow = ncol(x))
    unit_dirs <- apply(unit_dirs, 2, function(x) {
      tempnorm <- norm(x, type = "2")
      if (tempnorm == 0) { return(rep(0, length(x))) }
      else return(x / tempnorm)
    })
  }

  return(unit_dirs)
}


# Returns the index of the first feature vector that explains the 
# given percentage of variance
basic_fv_q <- function(fv, val = 99) {
  temp_scale <- matrixStats::colVars(fv)
  explained_scale <- temp_scale / sum(temp_scale) * 100
  cut <- which(cumsum(explained_scale) > val)[1]
  return(cut)
}


# Centering Kernel Matrix
center_Kmat <- function(K) {
  col_centered <- K - rep(1, nrow(K)) %*% t(colMeans(K))
  K_c <- col_centered - rowMeans(col_centered) %*% t(rep(1, ncol(K)))
  K_c <- (K_c + t(K_c)) / 2 # to prevent numerical instabilities
  return(K_c)
}


## Precision at top n (or top percentage) P@n
prec_n <- function(true_labels, outl, top_n = NULL, top_per = NULL) {
  if (is.null(top_n)) {
    if (is.null(top_per)) {
      stop("top_n or top_per must be specified")
    } else {
      top_n = floor(length(true_labels) * top_per)
    }
  }
  if (length(true_labels) != length(outl)) {
    stop("length of true_labels and outl must be the same")
  }
  tmp <- sort(outl, decreasing = TRUE, index.return = TRUE)
  top_n_idx <- tmp$ix[1:top_n]
  top_n_labels <- true_labels[top_n_idx]
  return(sum(top_n_labels) / top_n)
}


# Draw the cutoff line using log -> huberM and Qn
draw_cutoff <- function(outl, z = 0.99) {
  log_outl <- log(outl + 0.1)
  qn <- robustbase::Qn(log_outl)
  # HuberM for location
  m <- robustbase::huberM(log_outl, s = qn)$mu
  # cutoff
  cutoff <- exp(m + qn * qnorm(z)) - 0.1
  return(cutoff)
}


## Plot multiple outlyingness values
# takes input as KOD's output
plot_all_outl <- function(res, labels, ggsave_path = NULL) {
  point_ms <- res$scaled_outl_list$point
  two_points_ms <- res$scaled_outl_list$two_points
  basis_vector_ms <- res$scaled_outl_list$basis_vector
  random_ms <- res$scaled_outl_list$random
  miny = 0
  maxy = max(c(point_ms, two_points_ms, basis_vector_ms, random_ms,
               res$outl_list$final))

  # Combine all the outlyingness values in a single long-format data frame
  df <- data.frame(
    value = c(point_ms, two_points_ms, basis_vector_ms, random_ms,
              res$outl_list$final),
    type = factor(rep(c("One Point type", "Two Points type",
                        "Basis Vector type", "Random type", "Combined"),
                      times = c(length(point_ms), length(two_points_ms),
                                length(basis_vector_ms),
                                length(random_ms),
                                length(res$outl_list$final))),
                  levels = c("One Point type", "Two Points type",
                             "Basis Vector type", "Random type",
                             "Combined")),
    label = rep(labels, 5)
  )

  df$type = factor(df$type, levels = c("One Point type", "Two Points type", 
                                       "Basis Vector type", "Random type",
                                       "Combined"))

  # Optional: Create cutoff lines for each facet if different
  cutoffs <- data.frame(
    type = factor(c("One Point type", "Two Points type", "Basis Vector type",
                    "Random type", "Combined"),
                  levels = c("One Point type", "Two Points type", 
                             "Basis Vector type", "Random type", 
                             "Combined")),
    cutoff = c(
      draw_cutoff(point_ms),
      draw_cutoff(two_points_ms),
      draw_cutoff(basis_vector_ms),
      draw_cutoff(random_ms),
      res$cutoff
    ),
    linetype = c("dashed", "dashed", "dashed", "dashed", "solid")
  )

  # Plot with ggplot2 and facet_wrap
  # Main plot
  p <- ggplot(df, aes(x = (seq_along(value) - 1) %% 1000 + 1, y = value, 
                      color = factor(label))) +
    geom_point(size = 1.5) +
    scale_color_manual(values = c("0" = "#649C3D", "1" = "black"), 
                       guide = "none") +
    facet_wrap(~ type, nrow = 1, scales = "fixed") +
    geom_hline(data = cutoffs, aes(yintercept = cutoff, 
                                   linetype = linetype), color = "red", 
               show.legend = FALSE) +
    scale_linetype_manual(values = c("dashed" = "dashed", 
                                     "solid" = "solid"), guide = "none") +
    labs(y = "Outlyingness", x = "Index") +
    theme_minimal(base_size = 14) +
    theme(
      panel.background = element_rect(fill = "white", color = NA),  
      # panel area
      plot.background = element_rect(fill = "white", color = NA),   
      # entire image
      panel.grid = element_blank(),             
      # 1 - remove all grid lines
      panel.border = element_rect(color = "black", fill = NA, 
                                  linewidth = 0.8), 
      # 2 - add black border
      strip.text = element_text(size = 20),     
      # 3 - increase facet title size
      axis.text.x = element_text(size = 15),    
      # 4 - increase x-axis value size
      axis.text.y = element_text(size = 15),    
      # 5 - increase y-axis value size
      axis.title.x = element_text(size = 20),   
      # 6 - increase x-axis title size
      axis.title.y = element_text(size = 20),   
      # 7 - increase y-axis title size
      panel.spacing = unit(1, "lines")          
      # optional: spacing between panels
    )
  # p

  # 4 - Save as PNG
  ggsave(
    filename = ggsave_path,
    plot = p,
    width = 3500 / 200,   # = 17.5 inches
    height = 1000 / 200,  # = 5 inches
    dpi = 200,
    units = "in",
    pointsize = 15        # Note: `pointsize` is not directly used in 
    # ggplot2 but you can control it via base size
  )
}

# Utilities for Contour Plot ------------------------------------------

# Centers the new kernel matrix using the means of the original 
# kernel matrix
center_new_Kmat <- function(Kmat, Kmat_new) {
  n <- nrow(Kmat)
  m <- nrow(Kmat_new)
  col_means_K <- colMeans(Kmat)
  row_means_K_new <- rowMeans(Kmat_new)
  grand_mean_K <- mean(Kmat)
  mean_term1 <- matrix(col_means_K, nrow = m, ncol = n, byrow = TRUE)
  mean_term2 <- matrix(row_means_K_new, nrow = m, ncol = n)
  mean_term3 <- matrix(grand_mean_K, nrow = m, ncol = n)

  Kmat_new_c <- Kmat_new - mean_term1 - mean_term2 + mean_term3
  return(Kmat_new_c)
}


# Contour Plot - Calculating Outlyingness --------------------------------

# Note: this was an older version.
# The results are consistent with the main function (up to randomness),
# but the structure is slightly different and contains some unused 
# (in the final method) parameters.

KOD_gridcolor <- function(x,
                          kernel,
                          poly_degree = NULL,
                          dir_types = c("point", "basis_vector", 
                                        "random", "two_points"),
                          num_dirs = list(point = NULL, two_points = NULL, 
                                          basis_vector = NULL, random = 1000),
                          seed = NULL,
                          expvar = 99,
                          tol_fv = 1e-12,
                          tol_mad = 0.2,
                          labels = NULL,
                          center_for_point = "L1-median",
                          grid_data) {
  # Placeholders
  outl_list = list() # list of outlyingness values
  type_results = list() # list of detailed results for each direction type
  extra_list = NULL #
  result = list() # final result

  cutoff_mad = 0 # random-dir will change this

  # put random to first place (as it's used for MAD threshold)
  # NOTE: THIS METHOD ASSUMES THE USAGE OF RANDOM DIRECTION. ONE CANNOT 
  # SKIP THAT DIRTYPE
  if (dir_types[1] != "random") { 
    dir_types = c("random", dir_types[dir_types != "random"]) }


  # Generate seed if not provided
  if (is.null(seed)) {
    seed = ceiling(runif(1, 0, 1) * 10000)
  }

  # 1. Go feature space
  # Create kernel function
  if (kernel == "rbf") {
    bw = 1/(2*median(as.numeric(stats::dist(x))^2))
    kernel_function = kernlab::rbfdot(sigma = bw)
  } else if (kernel == "poly") {
    kernel_function = kernlab::polydot(degree = poly_degree)
  } else if (kernel == "linear") {
    kernel_function = kernlab::vanilladot()
  } else if (kernel == "rbf_nonrobust") {
    bw = 1/ncol(x)
    kernel_function = kernlab::rbfdot(sigma = bw)
  } else {
    stop("kernel not recognized")
  }

  # Obtain kernel matrix
  Kmat = kernlab::kernelMatrix(kernel_function, x)
  Kmat_c = center_Kmat(Kmat)
  # Obtain feature vectors
  transfv = classmap::makeFV(Kmat_c, precS = tol_fv)
  fv = transfv$Xf
  rm(Kmat_c)

  # Grid
  Kmat_grid = kernlab::kernelMatrix(kernel = kernel_function, 
                                    x = grid_data, y = x)
  Kmat_grid = center_new_Kmat(Kmat = Kmat, Kmat_new = Kmat_grid)
  grid_fv = classmap::makeFV(Kmat_grid, transfmat = transfv$transfmat)$Xf

  rm(Kmat, Kmat_grid, transfv)

  # CHECK EXPVAR
  if (expvar != 100) {
    cut_fv = basic_fv_q(fv, val = expvar)
    fv = fv[, 1:cut_fv]
    #
    grid_fv = grid_fv[, 1:cut_fv]
  }

  # 2. Run Modified Stahel-Donoho Outlyingness
  # Check number of directions
  if (is.null(num_dirs$point)) { num_dirs$point = nrow(fv) } 
  # use all points
  if (is.null(num_dirs$two_points)) { num_dirs$two_points = 2 * nrow(fv) } 
  # Subset of size 2*n
  if (is.null(num_dirs$basis_vector)) { num_dirs$basis_vector = ncol(fv) } 
  # use all basis vectors
  if (is.null(num_dirs$random)) { num_dirs$random = 1000 } 
  # use 1000 random directions

  # Run modified SDO for each direction type
  scaled_outl_list = list()
  scaled_outl_grid_list = list()

  for (dir_type in dir_types) {
    temp_numdir = num_dirs[[dir_type]]

    # Generate directions (generate subsets)
    unit_dirs = generate_unit_directions(x = fv,
                                         type = dir_type,
                                         num_dir = temp_numdir,
                                         center_for_point = center_for_point,
                                         seed = seed)

    ### Calculate outlyingness ###
    # Project data points to each direction
    proj_org = fv %*% unit_dirs # careful about t()
    # Center projection on each unit_dir by median
    proj = apply(proj_org, 2, function(x) {x - median(x)})
    # Truncation on denominator (MAD)
    mad_proj = apply(proj, 2, mad)
    if (dir_type == "random") {
      cutoff_mad = median(mad_proj) * tol_mad
    }

    small_mad_indices = which(mad_proj < cutoff_mad)
    mad_proj[small_mad_indices] = cutoff_mad
    # Calculate outlyingness
    all_outl = abs(sweep(proj, 2, mad_proj, FUN = "/"))
    outl = apply(all_outl, 1, max)

    # Save results
    type_results[[dir_type]]$outl = outl
    type_results[[dir_type]]$unit_dirs = unit_dirs
    type_results[[dir_type]]$proj = proj

    # cutoff
    log_outl = log(outl + 0.1)
    # Qn for scale
    qn = robustbase::Qn(log_outl)
    # HuberM for location
    m = robustbase::huberM(log_outl, s = qn)$mu
    # cutoff
    type_results[[dir_type]]$cutoff = exp(m + qn * qnorm(0.99)) - 0.1
    rm(log_outl, qn, m)

    # Scale by median
    scaled_outl_list[[dir_type]] = outl / median(outl)

    ### GRID RESULTS
    # Project grid points to each direction
    proj_grid = grid_fv %*% unit_dirs # careful about t()
    # Center projection on each unit_dir by median of training data
    median_projs = apply(proj_org, 2, median)
    proj_grid = sweep(proj_grid, 2, median_projs, FUN = "-")
    # Calculate outlyingness
    all_outl_grid = abs(sweep(proj_grid, 2, mad_proj, FUN = "/"))
    outl_grid = apply(all_outl_grid, 1, max)

    # Save results
    type_results[[dir_type]]$outl_grid = outl_grid

    scaled_outl_grid_list[[dir_type]] = outl_grid / median(outl)
  }

  # Combine outlyingness results
  outl_list[["final"]] = Reduce(pmax, scaled_outl_list)
  outl_list[["final_grid"]] = Reduce(pmax, scaled_outl_grid_list)

  # 3. Flagging outliers
  log_outl = log(outl_list[["final"]] + 0.1)
  # Qn for scale
  qn = robustbase::Qn(log_outl)
  # HuberM for location
  m = robustbase::huberM(log_outl, s = qn)$mu
  # cutoff
  cutoff = exp(m + qn * qnorm(0.99)) - 0.1
  # Flag outliers
  result$flagged_points = outl_list[["final"]] > cutoff

  if (!is.null(labels)) {
    # MCC (Matthews correlation coefficient)
    result$mcc = mltools::mcc(preds = c(outl_list[["final"]] > cutoff), 
                              actuals = (labels == 1))
    result$mcc = round(result$mcc, 3)
  }

  result$outl_list = outl_list
  result$type_results = type_results
  result$cutoff = cutoff

  result$extra_list = extra_list

  result$scaled_outl_grid_list = scaled_outl_grid_list
  result$scaled_outl_list = scaled_outl_list

  # Return result
  result$params = list(kernel = kernel,
                       poly_degree = poly_degree,
                       dir_types = dir_types,
                       num_dirs = num_dirs,
                       seed = seed,
                       tol_fv = tol_fv,
                       tol_mad = tol_mad,
                       labels = labels)

  return(result)
}





# Contour Plot - Generating The Plot --------------------------------------

# Define the function from the provided code
generate_heatmap <- function(grid_res,
                                  grid_data,
                                  data,
                                  y,
                                  color_scale = c("lightyellow", "lightyellow", "firebrick2"),
                                  ggsave_path = NULL,
                                  plot_the_data = FALSE,
                                  ggsave_path_data = NULL,
                                  ggsave_details = list(width = 10, height = 2.5, dpi = 600),
                                  ggpoint_details = list(pch = 1, size = 1, alpha = 1, stroke = 0.35),
                                  usefinal = TRUE,
                                  point_colors = c("black", "#649C3D"),
                                  put_cutoff = FALSE,
                                  asp_fixed = TRUE,
                                  title_text = NULL) {

  ## LOG
  grid_outl_point = log(grid_res$scaled_outl_grid_list$point + 0.1)
  grid_outl_two_points = log(grid_res$scaled_outl_grid_list$two_points + 0.1)
  grid_outl_basis_vector = log(grid_res$scaled_outl_grid_list$basis_vector + 0.1)
  grid_outl_random = log(grid_res$scaled_outl_grid_list$random + 0.1)
  grid_outl_final = log(grid_res$outl_list$final_grid + 0.1)

  ## LOG
  orig_outl_point = log(grid_res$scaled_outl_list$point + 0.1)
  orig_outl_two_points = log(grid_res$scaled_outl_list$two_points + 0.1)
  orig_outl_basis_vector = log(grid_res$scaled_outl_list$basis_vector + 0.1)
  orig_outl_random = log(grid_res$scaled_outl_list$random + 0.1)
  orig_outl_final = log(grid_res$outl_list$final + 0.1)

  # THE GRID
  gridplot_df_all = data.frame(x1 = grid_data[,1], x2 = grid_data[,2],
                               point = grid_outl_point,
                               two_points = grid_outl_two_points,
                               basis_vector = grid_outl_basis_vector,
                               random = grid_outl_random,
                               final = grid_outl_final)

  # Training (for visuals)
  origplot_df_all = data.frame(x1 = data[,1], x2 = data[,2],
                               point = orig_outl_point,
                               two_points = orig_outl_two_points,
                               basis_vector = orig_outl_basis_vector,
                               random = orig_outl_random,
                               final = orig_outl_final)


  # cutoff & other things
  cutoff_final = min(max(grid_outl_final) , log(grid_res$cutoff + 0.1)) # back to log
  min_overall = min(grid_outl_point, grid_outl_two_points, grid_outl_basis_vector, grid_outl_random, grid_outl_final)
  max_overall = max(grid_outl_point, grid_outl_two_points, grid_outl_basis_vector, grid_outl_random, grid_outl_final)

  middle_observation_point = quantile(orig_outl_point, 0.5)
  middle_observation_two_points = quantile(orig_outl_two_points, 0.5)
  middle_observation_basis_vector = quantile(orig_outl_basis_vector, 0.5)
  middle_observation_random = quantile(orig_outl_random, 0.5)
  middle_observation_final = quantile(orig_outl_final, 0.5)
  #
  normalized_cutoff_point <- (middle_observation_point - min_overall) / (max_overall - min_overall)
  normalized_cutoff_two_points <- (middle_observation_two_points - min_overall) / (max_overall - min_overall)
  normalized_cutoff_basis_vector <- (middle_observation_basis_vector - min_overall) / (max_overall - min_overall)
  normalized_cutoff_random <- (middle_observation_random - min_overall) / (max_overall - min_overall)
  normalized_cutoff_final <- (middle_observation_final - min_overall) / (max_overall - min_overall)
  #
  value_positions_point <- c(0, normalized_cutoff_point, 1)
  value_positions_two_points <- c(0, normalized_cutoff_two_points, 1)
  value_positions_basis_vector <- c(0, normalized_cutoff_basis_vector, 1)
  value_positions_random <- c(0, normalized_cutoff_random, 1)
  value_positions_final <- c(0, normalized_cutoff_final, 1)

  if (usefinal) {
    value_positions_point        = value_positions_final
    value_positions_two_points   = value_positions_final
    value_positions_basis_vector = value_positions_final
    value_positions_random       = value_positions_final
  }


  ## GGPLOTS
  gridplot_point <- ggplot() +
    geom_raster(data = gridplot_df_all, aes(x = x1, y = x2, fill = point)) +
    geom_point(data = origplot_df_all, aes(x = x1, y = x2),
               color = ifelse(y == 1, point_colors[1], point_colors[2]),
               pch = ggpoint_details$pch,
               size = ggpoint_details$size,
               alpha = ggpoint_details$alpha,
               stroke = ggpoint_details$stroke) +
    scale_fill_gradientn(colors = color_scale,
                         values = value_positions_point,
                         limits = c(min_overall, max_overall)) +
    theme_void() +
    labs(title = "One Point type") +
    labs(fill = "Outlyingness Value", color = "True Label") +
    theme(legend.position = "none") +
    theme(axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5))

  gridplot_two_points <- ggplot() +
    geom_raster(data = gridplot_df_all, aes(x = x1, y = x2, fill = two_points)) +
    geom_point(data = origplot_df_all, aes(x = x1, y = x2),
               color = ifelse(y == 1, point_colors[1], point_colors[2]),
               pch = ggpoint_details$pch,
               size = ggpoint_details$size,
               alpha = ggpoint_details$alpha,
               stroke = ggpoint_details$stroke) +
    scale_fill_gradientn(colors = color_scale,
                         values = value_positions_two_points,
                         limits = c(min_overall, max_overall)) +
    theme_void() +
    labs(title = "Two Points type") +
    labs(fill = "Outlyingness Value", color = "True Label") +
    theme(legend.position = "none") +
    theme(axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5))

  gridplot_basis_vector <- ggplot() +
    geom_raster(data = gridplot_df_all, aes(x = x1, y = x2, fill = basis_vector)) +
    geom_point(data = origplot_df_all, aes(x = x1, y = x2),
               color = ifelse(y == 1, point_colors[1], point_colors[2]),
               pch = ggpoint_details$pch,
               size = ggpoint_details$size,
               alpha = ggpoint_details$alpha,
               stroke = ggpoint_details$stroke) +
    scale_fill_gradientn(colors = color_scale,
                         values = value_positions_basis_vector,
                         limits = c(min_overall, max_overall)) +
    theme_void() +
    labs(title = "Basis Vector type") +
    labs(fill = "Outlyingness Value", color = "True Label") +
    theme(legend.position = "none") +
    theme(axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5))

  gridplot_random <- ggplot() +
    geom_raster(data = gridplot_df_all, aes(x = x1, y = x2, fill = random)) +
    geom_point(data = origplot_df_all, aes(x = x1, y = x2),
               color = ifelse(y == 1, point_colors[1], point_colors[2]),
               pch = ggpoint_details$pch,
               size = ggpoint_details$size,
               alpha = ggpoint_details$alpha,
               stroke = ggpoint_details$stroke) +
    scale_fill_gradientn(colors = color_scale,
                         values = value_positions_random,
                         limits = c(min_overall, max_overall)) +
    theme_void() +
    labs(title = "Random type") +
    labs(fill = "Outlyingness Value", color = "True Label") +
    theme(legend.position = "none") +
    theme(axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5))

  gridplot_final <- ggplot() +
    geom_raster(data = gridplot_df_all, aes(x = x1, y = x2, fill = final)) +
    geom_point(data = origplot_df_all, aes(x = x1, y = x2),
               color = ifelse(y == 1, point_colors[1], point_colors[2]),
               pch = ggpoint_details$pch,
               size = ggpoint_details$size,
               alpha = ggpoint_details$alpha,
               stroke = ggpoint_details$stroke) +
    scale_fill_gradientn(colors = color_scale,
                         values = value_positions_final,
                         limits = c(min_overall, max_overall)) +
    theme_void() +
    labs(title = "Combined") +
    labs(fill = "Outlyingness Value", color = "True Label") +
    theme(legend.position = "none") +
    theme(axis.text.x = element_blank(),
          axis.text.y = element_blank(),
          axis.ticks = element_blank(),
          axis.title.x = element_blank(),
          axis.title.y = element_blank(),
          plot.title = element_text(hjust = 0.5))

  if (put_cutoff) {
    gridplot_final = gridplot_final +
      geom_contour(data = gridplot_df_all, aes(x = x1, y = x2, z = final), breaks = c(cutoff_final), color = "black", linewidth = 0.8)
  }

  if (asp_fixed) {
    gridplot_point        = gridplot_point + coord_fixed()
    gridplot_two_points   = gridplot_two_points + coord_fixed()
    gridplot_basis_vector = gridplot_basis_vector + coord_fixed()
    gridplot_random       = gridplot_random + coord_fixed()
    gridplot_final        = gridplot_final + coord_fixed()
  }


  if (is.null(title_text)) {
    gridplot_all = grid.arrange(gridplot_point, gridplot_two_points, gridplot_basis_vector, gridplot_random, gridplot_final, ncol = 5)
  } else{
    gridplot_all = grid.arrange(gridplot_point, gridplot_two_points, gridplot_basis_vector, gridplot_random, gridplot_final, ncol = 5,
                                top = title_text)
  }

  ## Plot the original data, too

  if (plot_the_data) {
    dataplot_plot <- ggplot() +
      geom_raster(data = gridplot_df_all, aes(x = x1, y = x2), fill = "white") +
      geom_point(data = origplot_df_all, aes(x = x1, y = x2),
                 color = ifelse(y == 1, point_colors[1], point_colors[2]),
                 pch = ggpoint_details$pch,
                 size = ggpoint_details$size,
                 alpha = ggpoint_details$alpha,
                 stroke = ggpoint_details$stroke) +
      theme_void() +
      labs(title = "") +
      labs(fill = "", color = "") +
      theme(legend.position = "none") +
      theme(axis.text.x = element_blank(),
            axis.text.y = element_blank(),
            axis.ticks = element_blank(),
            axis.title.x = element_blank(),
            axis.title.y = element_blank(),
            plot.title = element_text(hjust = 0.5))


    if (asp_fixed) {dataplot_plot = dataplot_plot + coord_fixed()}


    if (!is.null(ggsave_path_data)) {
      # save it high quality
      ggsave(ggsave_path_data, plot = dataplot_plot, width = ggsave_details$width / 5,
             height = ggsave_details$height, dpi = ggsave_details$dpi)
    }

  }




  if (!is.null(ggsave_path)) {
    # save it high quality
    ggsave(ggsave_path, plot = gridplot_all, width = ggsave_details$width,
           height = ggsave_details$height, dpi = ggsave_details$dpi)
  }
  return(gridplot_all)
}








##########################
# Main Methodology --------------------------------------------------------
###########################

KOD <- function(x,
                                     kernel,
                                     poly_degree = NULL,
                                     doScale = FALSE,
                                     dir_types = c("random", "basis_vector", "point", "two_points"),
                                     num_dirs = list(two_points = 5000, random = 1000),
                                     seed = NULL,
                                     tol_fv = 1e-12,
                                     tol_mad = 0.2,
                                     expvar = 99, # explained variance: 0-100
                                     labels = NULL, # for obtaining performance metrics (not necessary)
                                     center_for_point = "L1-median",
                                     x_is_already_fv = FALSE) {

  # Main Function
  #
  #
  # x                           : input matrix
  # kernel                      : kernel function to use
  # poly_degree                 : degree of polynomial kernel
  # doScale                     : whether to scale the data. FALSE, TRUE or "mad"
  # dir_types                   : direction types for KO
  # num_dirs                    : number of directions for each type (ignored for dir_types point and basis_vector)
  # seed                        : seed for reproducibility
  # tol_fv                      : precS argument for classmap::makeFV function
  # tol_mad                     : coefficient for calculating the lower bound in denominator
  # expvar                      : percentage of explained variance to keep in feature space (0 to 100)
  # labels                      : true labels for performance metrics ## 0 is inliers, 1 is outliers
  # center_for_point            : center for point directions
  # x_is_already_fv             : whether the input is already in feature space
  #
  # browser()

  # check type of doScale
  if (doScale == "mad") {
    x <- scale(x, center = apply(x, 2, median), scale = apply(x, 2, mad))
  } else if (doScale == TRUE) {
    x <- scale(x)
  }


  # Placeholders
  outl_list <- list() # list of outlyingness values
  scaled_outl_list <- list() # list of normalized outlyingness values
  type_results <- list() # list of detailed results for each direction type
  result <- list() # final result

  cutoff_mad <- 0 # random-dir will change this


  # put random to first place (as it's used for MAD threshold)

  if (dir_types[1] != "random") { dir_types = c("random", dir_types[dir_types != "random"]) }

  # Generate seed if not provided
  if (is.null(seed)) {
    seed <- ceiling(runif(1, 0, 1) * 10000)
  }

  # 1. Go to feature space
  if (!x_is_already_fv) {
    # Create kernel function
    if (kernel == "rbf") {
      bw <- 1/(2*median((stats::dist(x))^2))
      kernel_function <- kernlab::rbfdot(sigma = bw)
    } else if (kernel == "poly") {
      kernel_function <- kernlab::polydot(degree = poly_degree)
    } else if (kernel == "linear") {
      kernel_function <- kernlab::vanilladot()
    } else {
      stop("Kernel not recognized. Only rbf, poly, and linear are supported.")
    }
    # Obtain kernel matrix
    Kmat <- center_Kmat(kernlab::kernelMatrix(kernel_function, x))
    # Obtain feature vectors
    fv <- classmap::makeFV(Kmat, precS = tol_fv)$Xf
  } else {
    fv <- x
  }

  # CHECK EXPVAR
  if (expvar != 100) {
    cut_fv <- basic_fv_q(fv, val = expvar)
    fv <- fv[, 1:cut_fv]
  }


  # 2. Run Modified Stahel-Donoho Outlyingness
  # Check and set number of directions
  num_dirs$point <- NULL # always uses all points
  num_dirs$basis_vector <- NULL # always uses all basis vectors
  if (is.null(num_dirs$two_points)) { num_dirs$two_points = 5 * nrow(fv) } # Subset of size 5*n
  if (is.null(num_dirs$random)) { num_dirs$random = 1000 } # use 1000 random directions

  # Run modified SDO for each direction type

  for (dir_type in dir_types) {
    temp_numdir <- num_dirs[[dir_type]]

    # Generate directions (generate subsets)
    unit_dirs <- generate_unit_directions(x = fv,
                                          type = dir_type,
                                          num_dir = temp_numdir,
                                          center_for_point = center_for_point,
                                          seed = seed)

    ### Calculate outlyingness ###
    # Project data points to each direction
    proj <- fv %*% unit_dirs
    # Center projection on each unit_dir by median
    proj <- apply(proj, 2, function(x) {x - median(x)})
    # Truncation on denominator (MAD)
    mad_proj <- apply(proj, 2, mad)

    if (dir_type == "random") {
      cutoff_mad <- median(mad_proj) * tol_mad
    }

    # Save outlyingness results on all directions before
    small_mad_indices <- which(mad_proj < cutoff_mad)
    mad_proj[small_mad_indices] <- cutoff_mad
    # Calculate outlyingness
    all_outl <- abs(sweep(proj, 2, mad_proj, FUN = "/"))
    outl <- apply(all_outl, 1, max)

    # Save results
    outl_list[[dir_type]] <- outl

    # Scale the outlyingness results by median
    scaled_outl_list[[dir_type]] <- outl / median(outl)

  }

  # Combine outlyingness results
  outl_list[["final"]] <- Reduce(pmax, scaled_outl_list)

  # 3. Flagging outliers
  # Obtain cutoff
  cutoff <- draw_cutoff(outl_list[["final"]], z = 0.99)
  # Flag outliers
  result$flagged_points <- outl_list[["final"]] > cutoff

  if (!is.null(labels)) {
    # MCC (Matthews correlation coefficient)
    result$mcc <- mltools::mcc(preds = c(outl_list[["final"]] > cutoff),
                               actuals = (labels == 1))
    result$mcc <- round(result$mcc, 3)
    # P@N (Precision at top N)
    result$patn <- prec_n(true_labels = labels,
                          outl = outl_list[["final"]],
                          top_n = sum(labels))
    result$patn <- round(result$patn, 3)
  }

  result$outl_list <- outl_list
  result$scaled_outl_list <- scaled_outl_list
  result$cutoff <- cutoff


  # Return result
  result$params <- list(kernel = kernel,
                        poly_degree = poly_degree,
                        dir_types = dir_types,
                        num_dirs = num_dirs,
                        seed = seed,
                        tol_fv = tol_fv,
                        tol_mad = tol_mad,
                        cutoff_mad = cutoff_mad,
                        labels = labels)

  return(result)
}







