function [f, g, h] = Funct_ObjGradHessian_V(X, Xt, V, R, zeta, ZeroOut, U)
%==========================================================================
% This function computes the function value, Euclidean gradient and Hessian
% of the V objective function in computing PHD (assuming smoothness)
%------------------------ Input Variables ---------------------------------
% X               - design matrix
% Xt              - transpose of X
% V               - current point V 
% R               - residual matrix
% zeta            - annealing parameter for the heating oprocess (inverse cooling), which controls
%                   the steepness of the function in approximating the sign function 
% ZeroOut         - field mat: a binary matrix of the same size of B. 0 means the position should be fixed at 0
% U               - direction U of interest (only used in calculating h) 
%------------------------ Output Variables ---------------------------------
% f               - objective function value
% g               - euclidean gradient
% h 	          - euclidean hessian along U
%==========================================================================
% To calculate diag( X * V * R'), the following two ways are not efficient
%Xdiag = diag(X*V*Rt);        % This is not efficient!
%Xdiag = diagonal(X,V,Rt);    % faster than the first one but not efficient enough.

[p, m] = size(V);

if ~exist('U', 'var') || isempty(U)
    U = 0;
end

if ~exist('ZeroOut', 'var')
    ZeroOut =[];
end
ZeroOutEffective = exist('ZeroOut', 'var') && ~isempty(ZeroOut);

if ~ZeroOutEffective || (ZeroOutEffective && strcmpi(ZeroOut.VMethod, 'constrained'))
    Xdiag =  sum((X*V).*R, 2); % diag( X * V * R')
    switch nargout
        case 1
            [f1] = aux_Phi(Xdiag, zeta);
            f = sum(f1);
        case 2 % gradient required
            [f1,g1] = aux_Phi(Xdiag, zeta);
            f = sum(f1);
            g = Xt*bsxfun(@times, R, g1); %X.'*diag(g1)*R;
        case 3 % Hessian required
            [f1,g1,h1] = aux_Phi(Xdiag, zeta);
            f = sum(f1);
            g = Xt*bsxfun(@times, R, g1);
            h = Xt*bsxfun(@times, R, h1.*sum((X*U).*R, 2)); %Only require the diagonal elements of X*U*R.'
    end
    
elseif strcmpi(ZeroOut.VMethod, 'penalized')
        if isempty(ZeroOut.lambda_V), error('No penalty parameter provided to enforce the zeros in V'), end
        
        Xdiag =  sum((X*V).*R, 2); % diag( X * V * R')
        switch nargout
            case {0, 1}
                [f1] = aux_Phi(Xdiag, zeta) ;
                f = sum(f1) + ZeroOut.lambda_V * norm(V .* (~ZeroOut.mat), 'fro')^ 2 / 2;
            case 2 % gradient required
                [f1,g1] = aux_Phi(Xdiag, zeta);
                f = sum(f1) + ZeroOut.lambda_V * norm(V .* (~ZeroOut.mat), 'fro')^ 2 / 2;
                g = Xt*bsxfun(@times, R, g1) + ZeroOut.lambda_V * (V .* (~ZeroOut.mat)); %X.'*diag(g1)*R;
            case 3 % Hessian required
                [f1,g1,h1] = aux_Phi(Xdiag, zeta);
                f = sum(f1) + ZeroOut.lambda_V * norm(V .* (~ZeroOut.mat), 'fro')^ 2 / 2;
                g = Xt*bsxfun(@times, R, g1) + ZeroOut.lambda_V * (V .* (~ZeroOut.mat)); %X.'*diag(g1)*R;
                h = Xt*bsxfun(@times, R, h1.*sum((X*U).*R, 2)) + ZeroOut.lambda_V *(U .* (~ZeroOut.mat)); 
        end
    else
        error('Not implemented')
    
end


end

